
suppressMessages(library(RKernel))
options(width=200)

# ## Preparing the data
# The data we are using are in the file "ESS-relig-CHES.RData".

suppressMessages(library(memisc))
library(magrittr)


load(file="ESS-relig-CHES.RData")


load(file="perc-denom-country-wide.RData")

# We restrict our analysis to the denominationally mixed countries in Western
# Europe, that is, Germany, the Netherlands, Switzerland, and the United Kingdom.

ESS_work_west <- subset(ESS_denom_CHES,
    country2 %in% c(
        "Austria",
        "Belgium/Wallonia","Belgium/Flanders","Belgium/Brussels",
        "Denmark",
        "France",
        "Germany (West)","Germany/Bavaria",
        "Ireland",
        "Italy",
        "Luxembourg",
        "Netherlands",
        "Norway",
        "Portugal",
        "Spain/Castilia","Spain/Catalunia","Spain/Basque",
        "Sweden",
        "Switzerland",
        "United Kingdom/England",
        "United Kingdom/Scotland",
        "United Kingdom/Wales"
    ) & !is.na(sociallifestyle),
    select=c(CHES_id,
             year,
             essround,
             denom,
             oesch8,
             wgt,
             id,
             country2,
             lrecon,
             galtan,
             spendvtax,
             deregulation,
             redistribution,
             civlib_laworder,
             sociallifestyle,
             religious_principles,
             immigration))


ESS_work_west %<>% as.data.frame()

# The information about religious denomination appears to be incorrect or misleading 
# in France until 2005 and in the United Kingdom from 2004 to 2007 and after 2020.

options(contrasts=c("contr.sum","contr.sum"))


ESS_work_west %<>% within({
    Country <- recode(as.character(country2),
                      c("Belgium/Wallonia","Belgium/Flanders","Belgium/Brussels") -> "Belgium",
                      c("Germany (West)","Germany/Bavaria") -> "Germany (West)",
                      c("Spain/Castilia","Spain/Catalunia","Spain/Basque") -> "Spain",
                      c("United Kingdom/England","United Kingdom/Scotland","United Kingdom/Wales") 
                          -> "United Kingdom",
                      copy=TRUE)
    Country <- factor(Country,levels=sort(levels(Country)))
    denom[Country=="France" & year <= 2005 | 
         Country=="United Kingdom" & year >= 2004 & year <= 2007] <- NA
    relmemb <- recode(as.numeric(denom),
                      "Christian"     <- 1:4,
                      "Non-Christian" <- 5,
                      "No religion"   <- 6
                     )
})


ESS_work_west <- merge(ESS_work_west, perc.denom.country.wide,
                       by.x = 'Country',
                       by.y = 'country',
                       all.x = TRUE,
                       all.y = FALSE)


ESS_work_west %<>% within({
    foreach(x = c(lrecon,
                  galtan,
                  spendvtax,
                  deregulation,
                  redistribution,
                  civlib_laworder,
                  sociallifestyle,
                  religious_principles,
                  immigration),
            y = c(lrecon.sq,
                  galtan.sq,
                  spendvtax.sq,
                  deregulation.sq,
                  redistribution.sq,
                  civlib_laworder.sq,
                  sociallifestyle.sq,
                  religious_principles.sq,
                  immigration.sq),
            {
                x <- (x-5)/10
                y <- x^2
            })
    oesch8 <- factor(oesch8,levels=levels(oesch8)[c(3,1:2,4:8)]) # Production workers as baseline
    country <- factor(country2) # Drop countries not included in this subset
    time <- (year - 2010)/10
    contrasts(relmemb) <- contr("sum", base = 1)
    contrasts(oesch8) <- contr("sum", base = 1)
    Secularization <- (No_religion - 50)/100
    Cath_Prot <- (Cath_Prot - 50)/100
    })

# ## Model estimation

library(mclogit)


model_soclife_full <- mclogit(wgt|id~sociallifestyle.sq+
                                immigration.sq+
                                lrecon.sq+
                                sociallifestyle/(oesch8+relmemb)*time+
                                sociallifestyle/(relmemb*Secularization*time)+
                                sociallifestyle/(relmemb*Cath_Prot*time)+
                                immigration/(oesch8+relmemb)*time+
                                lrecon/(oesch8+relmemb)*time
                                -time,
                                random=~1|CHES_id/essround,
                                data=ESS_work_west,
                                 trace=TRUE,
                                estimator="REML"
                                )


# ## Wald tests

source("Wald.R")


(Wald_soclife_full <- 
 Wald(model_soclife_full,
     list(~sociallifestyle.sq,
          ~sociallifestyle,
          ~sociallifestyle:time,
          ~sociallifestyle:relmemb,
          ~sociallifestyle:relmemb:time,
          ~sociallifestyle:Secularization,
          ~sociallifestyle:relmemb:Secularization,
          ~sociallifestyle:time:Secularization,
          ~sociallifestyle:relmemb:time:Secularization,
          ~sociallifestyle:Cath_Prot,
          ~sociallifestyle:relmemb:Cath_Prot,
          ~sociallifestyle:time:Cath_Prot,
          ~sociallifestyle:relmemb:time:Cath_Prot
           )))


Wald_soclife_full %>%
        dimrename(          
        "sociallifestyle.sq" = "Social Lifestyle squared",
        "sociallifestyle" = "Social Lifestyle",
        "sociallifestyle:relmemb" = "$\\times$ religious membership",
        "sociallifestyle:time" = "$\\times$ time",
        "sociallifestyle:relmemb:time" = "$\\times$ time $\\times$ religious membership",
        "sociallifestyle:Secularization" = "$\\times$ secularization",
        "sociallifestyle:relmemb:Secularization" = "$\\times$ secularization $\\times$ religious membership",
        "sociallifestyle:time:Secularization" = "$\\times$ secularization $\\times$ time",            
        "sociallifestyle:relmemb:time:Secularization" = "$\\times$ secularization $\\times$ religious membership $\\times$ time",            
        "sociallifestyle:Cath_Prot" = "$\\times$ Reformation",
        "sociallifestyle:time:Cath_Prot" = "$\\times$ Reformation $\\times$ time",
        "sociallifestyle:relmemb:Cath_Prot" = "$\\times$ Reformation $\\times$ religious membership",
        "sociallifestyle:relmemb:time:Cath_Prot" = "$\\times$ Reformation $\\times$ religious membership $\\times$ time"            
        ) %>% rename(
            "Chisq"="$W$",
            "Pr(>Chisq)"="$p$-value") %>%
        toLatex(digits=c(1,0,3)) %>% writeLines(con="relmemb-context-soclife/Wald-relmemb-context-soclife-full.tex")


model_soclife <- mclogit(wgt|id~immigration.sq+
                                lrecon.sq+
                                sociallifestyle.sq +
                                sociallifestyle/(oesch8+relmemb)*time+
                                sociallifestyle/(relmemb*Secularization)+
                                immigration/(oesch8+relmemb)*time+
                                lrecon/(oesch8+relmemb)*time
                                -time,
                                random=~1|CHES_id/essround,
                                data=ESS_work_west,
                                 trace=TRUE,
                                estimator="REML"
                                )



(Wald_soclife <- 
 Wald(model_soclife,
     list(~sociallifestyle.sq,
          ~sociallifestyle,
          ~sociallifestyle:time,
          ~sociallifestyle:relmemb,
          ~sociallifestyle:relmemb:time,
          ~sociallifestyle:Secularization,
          ~sociallifestyle:relmemb:Secularization
           )))


Wald_soclife %>%
        dimrename(          
        "sociallifestyle.sq" = "Social Lifestyle squared",
        "sociallifestyle" = "Social Lifestyle",
        "sociallifestyle:relmemb" = "$\\times$ religious membership",
        "sociallifestyle:time" = "$\\times$ time",
        "sociallifestyle:relmemb:time" = "$\\times$ time $\\times$ religious membership",
        "sociallifestyle:Secularization" = "$\\times$ secularization",
        "sociallifestyle:relmemb:Secularization" = "$\\times$ secularization $\\times$ religious membership"
        ) %>% rename(
            "Chisq"="$W$",
            "Pr(>Chisq)"="$p$-value") %>%
        toLatex(digits=c(1,0,3)) %>% writeLines(con="relmemb-context-soclife/Wald-relmemb-context-soclife.tex")


save(model_soclife,model_soclife_full,file="model-relmemb-soclife-context.RData")

